<?php
/**
 * Exit if accessed directly
 */
defined( 'ABSPATH' ) || exit( 'دسترسی غیر مجاز!' );

class WAST_Send_Email {

	/**
	 * WAST_Send_Email constructor.
	 */
	public function __construct() {
		add_filter( 'wp_mail_from_name', function ( $original_email_from ) {
			return get_bloginfo( 'name' );
		} );
	}

	/**
	 * Send email
	 *
	 * @param $to
	 * @param $ticket_id
	 *
	 * @return bool
	 */
	public function send( $to, $ticket_id ) {
		if ( ! $to || ! is_email( $to ) || ! intval( $ticket_id ) ) {
			return false;
		}

		$active  = wast_get_option( 'active-notifications' );
		$subject = wast_get_option( 'email-subject' );
		$from    = wast_get_option( 'email-sender' );

		if ( ! $active ) {
			return false;
		}

		wp_mail( $to, $subject, $this->get_message( $to, $ticket_id ), $this->get_headers( $from ) );
	}

	/**
	 * Get message body
	 *
	 * @param $to
	 * @param $ticket_id
	 *
	 * @return mixed
	 */
	private function get_message( $to, $ticket_id ) {
		ob_start();
		include( WAST_TPL_PATH . 'email/tpl-email.php' );
		$message = ob_get_clean();

		$find = array(
			'##site-logo##',
			'##recipient-name##',
			'##sender-name##',
			'##date##',
			'##product-title##',
			'##ticket-title##',
			'##ticket-url##',
			'##footer-text##',
			'##site-url##',
		);

		$logo = wast_get_option( 'site-logo' ) ? wast_get_option( 'site-logo' ) : '';

		$ticket = get_post( $ticket_id );

		$recipient              = get_user_by( 'email', $to );
		$recipient_id           = $recipient->ID;
		$recipient_display_name = $recipient->display_name;

		$sender_id    = get_current_user_id();
		$user         = get_userdata( $sender_id );
		$display_name = $user->display_name;

		$product_id    = get_post_meta( $ticket_id, '_wast_ticket_product_id', true );
		$product_title = $product_id;
		if ( is_numeric( $product_id ) ) {
			$product_title = get_the_title( $product_id );
		}
		if ( ! $product_title || empty( $product_title ) ) {
			$message = str_replace( 'class="p-title"', 'class="p-title" style="display:none;"', $message );
		}

		$ticket_title = $ticket->post_title;

		$ticket_url = $this->get_ticket_url( $recipient_id, $ticket_id );

		$footer = wast_get_option( 'email-footer-text' ) ? wast_get_option( 'email-footer-text' ) : '';

		$replace = array(
			$logo,
			$recipient_display_name,
			$display_name,
			date_i18n( 'Y/m/d H:i', time() ),
			$product_title,
			$ticket_title,
			'<a href="' . $ticket_url . '" target="_blank" style="text-decoration: none; color: #fff; background: #ffa502; padding: 7px 15px; border-radius: 31px;">پاسخ به تیکت</a>',
			$footer,
			'<a href="' . esc_url( home_url() ) . '" target="_blank" style="text-decoration: none;">' . get_bloginfo( 'name' ) . '</a>',
		);

		return str_replace( $find, $replace, $message );
	}

	/**
	 * Header for nice email
	 *
	 * @param $from
	 *
	 * @return string
	 */
	private function get_headers( $from ) {
		$headers = "From: " . strip_tags( $from ) . "\r\n";
		$headers .= "MIME-Version: 1.0\r\n";
		$headers .= "Content-Type: text/html; charset=UTF-8\r\n";

		return $headers;
	}

	/**
	 * Retrieve ticket reply url for admin, seller and user
	 *
	 * @param $recipient_id
	 * @param $ticket_id
	 *
	 * @return null|string
	 */
	private function get_ticket_url( $recipient_id, $ticket_id ) {
		if ( ! $ticket_id ) {
			return null;
		}

		$url = wast_get_ticket_page_url( $recipient_id );

		if ( is_admin() ) {
			$ticket_url = add_query_arg(
				array(
					'action'    => 'reply',
					'ticket_id' => $ticket_id
				), $url );
		} else {
			if ( wast_is_user_seller( $recipient_id ) ) {
				$ticket_url = add_query_arg(
					array(
						'action'    => 'reply',
						'ticket_id' => $ticket_id
					), $url );
			} else {
				$ticket_url = esc_url(
					add_query_arg( array(
						'action' => 'edit',
						'post'   => $ticket_id
					), admin_url( 'post.php' ) ) );
			}
		}

		return $ticket_url;
	}
}